﻿// Tube Link Fixer Extension - Content Script for YouTube
// Enhanced link extraction from video descriptions

console.log('[Tube Link Fixer] Content script initializing...');

// Listen for messages from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log('[Tube Link Fixer] Received message:', message);

    if (message.action === 'extractLinks') {
        // First try to expand the description
        expandDescription().then(() => {
            // Wait a bit for content to load after expanding
            setTimeout(() => {
                const links = extractDescriptionLinks();
                console.log('[Tube Link Fixer] Extracted links:', links);
                sendResponse({ links, success: true });
            }, 500);
        });
        return true; // Keep channel open for async response
    }

    if (message.action === 'ping') {
        sendResponse({ success: true, message: 'Content script is active' });
        return true;
    }
});

// Try to expand the description if it's collapsed
async function expandDescription() {
    // Look for "Show more" or "...more" button
    const expandButtons = [
        'tp-yt-paper-button#expand',
        '#expand',
        '#description-inline-expander #expand',
        'ytd-text-inline-expander #expand',
        '[aria-label="Show more"]',
        '.more-button',
        '#snippet #expand'
    ];

    for (const selector of expandButtons) {
        const button = document.querySelector(selector);
        if (button && button.offsetParent !== null) {
            console.log('[Tube Link Fixer] Found expand button, clicking...');
            button.click();
            // Wait for expansion animation
            await new Promise(resolve => setTimeout(resolve, 300));
            return true;
        }
    }

    return false;
}

// Extract all links from the video description
function extractDescriptionLinks() {
    const links = new Set();

    console.log('[Tube Link Fixer] Starting link extraction...');

    // Primary description selectors (in order of preference)
    const descriptionSelectors = [
        '#description-inline-expander',
        'ytd-text-inline-expander[slot="content"]',
        '#description ytd-text-inline-expander',
        '#structured-description',
        '#description-inner',
        '#description',
        'ytd-watch-metadata #description',
        '#meta-contents #description',
        '.ytd-video-secondary-info-renderer #description'
    ];

    // Try each selector
    for (const selector of descriptionSelectors) {
        const container = document.querySelector(selector);
        if (container) {
            console.log(`[Tube Link Fixer] Found description container: ${selector}`);
            extractLinksFromElement(container, links);
        }
    }

    // Also check for attributed strings (YouTube's formatted text)
    const attributedStrings = document.querySelectorAll(
        '#description yt-attributed-string, ' +
        '#description-inner yt-attributed-string, ' +
        'ytd-text-inline-expander yt-attributed-string'
    );

    attributedStrings.forEach(el => {
        extractLinksFromElement(el, links);
    });

    // Check structured description sections
    const structuredSections = document.querySelectorAll(
        'ytd-structured-description-content-renderer'
    );
    structuredSections.forEach(section => {
        extractLinksFromElement(section, links);
    });

    const result = Array.from(links);
    console.log(`[Tube Link Fixer] Total unique links found: ${result.length}`);

    return result;
}

// Extract links from a DOM element
function extractLinksFromElement(element, linksSet) {
    if (!element) return;

    // Get all anchor tags
    const anchors = element.querySelectorAll('a');

    anchors.forEach(anchor => {
        let href = anchor.href;

        if (!href) return;

        // Handle YouTube redirect URLs
        if (href.includes('youtube.com/redirect') || href.includes('/redirect?')) {
            try {
                const url = new URL(href);
                const actualUrl = url.searchParams.get('q') || url.searchParams.get('url');
                if (actualUrl) {
                    href = actualUrl;
                }
            } catch (e) {
                console.log('[Tube Link Fixer] Error parsing redirect:', e);
            }
        }

        // Validate and add the link
        if (isValidExternalLink(href)) {
            linksSet.add(href);
            console.log(`[Tube Link Fixer] Found link: ${href}`);
        }
    });

    // Also look for links in text content (URLs that might not be anchor tags)
    const textContent = element.textContent || '';
    const urlRegex = /https?:\/\/[^\s<>"{}|\\^`\[\]]+/g;
    const textUrls = textContent.match(urlRegex) || [];

    textUrls.forEach(url => {
        // Clean up the URL (remove trailing punctuation)
        let cleanUrl = url.replace(/[.,;:!?)]+$/, '');

        if (isValidExternalLink(cleanUrl)) {
            linksSet.add(cleanUrl);
        }
    });
}

// Check if a link is a valid external URL
function isValidExternalLink(url) {
    if (!url || typeof url !== 'string') return false;

    try {
        const parsed = new URL(url);

        // Must be http or https
        if (parsed.protocol !== 'http:' && parsed.protocol !== 'https:') {
            return false;
        }

        // Skip YouTube internal links
        const hostname = parsed.hostname.toLowerCase();
        if (
            hostname.includes('youtube.com') ||
            hostname.includes('youtu.be') ||
            hostname.includes('ytimg.com') ||
            hostname.includes('googlevideo.com') ||
            hostname.includes('ggpht.com') ||
            hostname === 'www.google.com' ||
            hostname === 'google.com' ||
            hostname === 'accounts.google.com'
        ) {
            return false;
        }

        // Skip browser internal URLs
        if (
            parsed.protocol === 'javascript:' ||
            parsed.protocol === 'chrome:' ||
            parsed.protocol === 'chrome-extension:' ||
            parsed.protocol === 'about:' ||
            parsed.protocol === 'data:'
        ) {
            return false;
        }

        // Skip empty or hash-only links
        if (!parsed.hostname || parsed.hostname.length < 3) {
            return false;
        }

        return true;

    } catch (e) {
        return false;
    }
}

// Wait for the page to be ready
function waitForDescription(callback, maxAttempts = 30) {
    let attempts = 0;

    const check = () => {
        attempts++;
        const description = document.querySelector('#description, #description-inline-expander');

        if (description) {
            console.log('[Tube Link Fixer] Description element found');
            callback(true);
        } else if (attempts < maxAttempts) {
            setTimeout(check, 500);
        } else {
            console.log('[Tube Link Fixer] Description not found after max attempts');
            callback(false);
        }
    };

    check();
}

// Initialize
function init() {
    console.log('[Tube Link Fixer] Content script loaded on YouTube video page');

    waitForDescription((found) => {
        if (found) {
            console.log('[Tube Link Fixer] Ready to scan - description detected');
        } else {
            console.log('[Tube Link Fixer] Warning: Could not find description element');
        }
    });
}

// Run initialization
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}

// Also reinitialize when YouTube navigates (SPA navigation)
let lastUrl = location.href;
new MutationObserver(() => {
    if (location.href !== lastUrl) {
        lastUrl = location.href;
        console.log('[Tube Link Fixer] URL changed, reinitializing...');
        setTimeout(init, 1000);
    }
}).observe(document.body, { subtree: true, childList: true });
