﻿// Tube Link Fixer Extension - Background Service Worker
// Enhanced link checking with retry logic and proper status detection

console.log('[Tube Link Fixer] Background service worker started');

// Social media and known platforms that may return false errors
const SOCIAL_MEDIA_DOMAINS = [
    'twitter.com', 'x.com', 'facebook.com', 'fb.com', 'instagram.com',
    'linkedin.com', 'tiktok.com', 'pinterest.com', 'reddit.com',
    'discord.com', 'discord.gg', 'twitch.tv', 'tumblr.com',
    'snapchat.com', 'whatsapp.com', 'telegram.org', 't.me',
    'medium.com', 'substack.com', 'patreon.com', 'ko-fi.com',
    'gumroad.com', 'etsy.com', 'amazon.com', 'amzn.to', 'amzn.com',
    'spotify.com', 'open.spotify.com', 'apple.com', 'music.apple.com',
    'soundcloud.com', 'bandcamp.com', 'github.com', 'gitlab.com',
    'notion.so', 'notion.site', 'trello.com', 'asana.com',
    'dropbox.com', 'drive.google.com', 'docs.google.com'
];

// Check if URL is from a social media or known platform
function isSocialMediaOrKnownPlatform(url) {
    try {
        const hostname = new URL(url).hostname.toLowerCase();
        return SOCIAL_MEDIA_DOMAINS.some(domain =>
            hostname === domain || hostname.endsWith('.' + domain)
        );
    } catch {
        return false;
    }
}

// Listen for messages
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log('[Tube Link Fixer] Background received:', message.action || message.type);

    if (message.action === 'checkLink') {
        checkLinkStatus(message.url)
            .then(result => {
                console.log(`[Tube Link Fixer] Link check result for ${message.url}:`, result.status);
                sendResponse(result);
            })
            .catch(error => {
                console.error('[Tube Link Fixer] Link check error:', error);
                sendResponse({ url: message.url, status: 'broken', code: 'ERROR', error: error.message });
            });
        return true; // Keep channel open for async response
    }

    if (message.type === 'Tube Link Fixer_AUTH_SUCCESS') {
        console.log('[Tube Link Fixer] Auth success, storing credentials...');
        chrome.storage.sync.set({
            authToken: message.token,
            userInfo: message.userInfo
        }, () => {
            console.log('[Tube Link Fixer] Credentials stored');
        });
        sendResponse({ success: true });
        return true;
    }

    if (message.action === 'syncResults') {
        syncResultsToServer(message.results).then(sendResponse);
        return true;
    }
});

// Check link status with retry logic
async function checkLinkStatus(url, retryCount = 0) {
    const maxRetries = 2;
    const isSocial = isSocialMediaOrKnownPlatform(url);

    const result = {
        url: url,
        status: 'unknown',
        code: null,
        error: null
    };

    try {
        // First try: Simple fetch with mode that allows status reading
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 15000);

        try {
            // Try a regular fetch first (may fail due to CORS)
            const response = await fetch(url, {
                method: 'GET',
                signal: controller.signal,
                redirect: 'follow',
                mode: 'cors',
                credentials: 'omit',
                cache: 'no-cache',
            });

            clearTimeout(timeoutId);

            const statusCode = response.status;
            result.code = statusCode;

            if (statusCode >= 200 && statusCode < 300) {
                result.status = 'valid';
            } else if (statusCode >= 300 && statusCode < 400) {
                result.status = 'redirect';
            } else if (statusCode >= 400 && statusCode < 500) {
                if (isSocial && (statusCode === 403 || statusCode === 429 || statusCode === 405)) {
                    // Social media often blocks automated requests
                    result.status = 'valid'; // Assume working
                    result.code = 'BLOCKED';
                } else if (statusCode === 404 || statusCode === 410) {
                    result.status = 'broken';
                } else {
                    result.status = isSocial ? 'valid' : 'broken';
                }
            } else if (statusCode >= 500) {
                if (retryCount < maxRetries) {
                    await new Promise(resolve => setTimeout(resolve, 1000 * (retryCount + 1)));
                    return checkLinkStatus(url, retryCount + 1);
                }
                result.status = 'broken';
            }

            return result;

        } catch (corsError) {
            clearTimeout(timeoutId);

            // CORS error - try no-cors mode (limited info but can detect if request completes)
            return await checkLinkWithNoCors(url, isSocial, retryCount);
        }

    } catch (error) {
        if (error.name === 'AbortError') {
            // Timeout
            if (retryCount < maxRetries) {
                await new Promise(resolve => setTimeout(resolve, 1000));
                return checkLinkStatus(url, retryCount + 1);
            }
            result.status = isSocial ? 'valid' : 'broken';
            result.code = 'TIMEOUT';
            result.error = 'Request timed out';
        } else {
            // Network error
            if (retryCount < maxRetries && isSocial) {
                await new Promise(resolve => setTimeout(resolve, 1000));
                return checkLinkStatus(url, retryCount + 1);
            }
            result.status = isSocial ? 'valid' : 'broken';
            result.code = 'ERROR';
            result.error = error.message;
        }

        return result;
    }
}

// Check link with no-cors mode (fallback)
async function checkLinkWithNoCors(url, isSocial, retryCount = 0) {
    const result = {
        url: url,
        status: 'unknown',
        code: null,
        error: null
    };

    try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000);

        // no-cors fetch - we can't read status but can detect if request completes
        const response = await fetch(url, {
            method: 'HEAD', // Try HEAD first (faster)
            signal: controller.signal,
            mode: 'no-cors',
            credentials: 'omit',
            redirect: 'follow'
        });

        clearTimeout(timeoutId);

        // With no-cors, response.type will be 'opaque' if successful
        if (response.type === 'opaque' || response.ok) {
            result.status = 'valid';
            result.code = 200; // Assumed
            return result;
        }

        // If we got here, something worked
        result.status = 'valid';
        result.code = 200;
        return result;

    } catch (error) {
        // Try GET as last resort
        try {
            const controller2 = new AbortController();
            const timeoutId2 = setTimeout(() => controller2.abort(), 10000);

            await fetch(url, {
                method: 'GET',
                signal: controller2.signal,
                mode: 'no-cors',
                credentials: 'omit',
                redirect: 'follow'
            });

            clearTimeout(timeoutId2);

            // If we got here without error, assume it works
            result.status = 'valid';
            result.code = 200;
            return result;

        } catch (fallbackError) {
            if (fallbackError.name === 'AbortError') {
                result.status = isSocial ? 'valid' : 'broken';
                result.code = 'TIMEOUT';
            } else {
                // Actual network failure
                result.status = 'broken';
                result.code = 'ERROR';
                result.error = fallbackError.message;
            }
            return result;
        }
    }
}

// Sync scan results to Tube Link Fixer server
async function syncResultsToServer(results) {
    try {
        const data = await chrome.storage.sync.get(['authToken']);
        if (!data.authToken) {
            return { success: false, error: 'Not authenticated' };
        }

        const response = await fetch('https://tube-link-fixer.vercel.app/api/extension/sync', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${data.authToken}`
            },
            body: JSON.stringify({ results })
        });

        if (!response.ok) {
            throw new Error('Sync failed');
        }

        return { success: true };

    } catch (error) {
        return { success: false, error: error.message };
    }
}

// Listen for tab updates to check auth status from LocalStorage
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url) {
        // Check if this is our auth page
        if (tab.url.includes('/extension-auth')) {
            // Start polling for auth data immediately
            pollForAuthData(tabId, 0);
        }
    }
});

// Poll for auth data with retries
async function pollForAuthData(tabId, attempt) {
    const maxAttempts = 30; // Poll for up to 30 seconds
    const pollInterval = 1000; // Check every 1 second

    if (attempt >= maxAttempts) {
        console.log('[Tube Link Fixer] Auth polling timed out');
        return;
    }

    try {
        const results = await chrome.scripting.executeScript({
            target: { tabId: tabId },
            func: () => {
                const authData = localStorage.getItem('Tube Link Fixer_extension_auth');
                if (authData) {
                    try {
                        const parsed = JSON.parse(authData);
                        // Check if auth data is fresh (within last 5 minutes)
                        if (parsed.timestamp && Date.now() - parsed.timestamp < 5 * 60 * 1000) {
                            // Remove the data so it's not picked up again
                            localStorage.removeItem('Tube Link Fixer_extension_auth');
                            return parsed;
                        }
                    } catch (e) {
                        console.error('Error parsing auth data:', e);
                    }
                }
                return null;
            }
        });

        if (results && results[0] && results[0].result) {
            const authData = results[0].result;
            console.log('[Tube Link Fixer] Auth data found via polling, storing credentials...');

            // Store the credentials
            await chrome.storage.sync.set({
                authToken: authData.token,
                userInfo: authData.userInfo
            });

            console.log('[Tube Link Fixer] Credentials stored successfully');

            // Broadcast to all extension views that auth succeeded
            chrome.runtime.sendMessage({
                type: 'Tube Link Fixer_AUTH_SUCCESS',
                token: authData.token,
                userInfo: authData.userInfo
            }).catch(() => {
                // Popup might not be open, that's okay
            });

            // Close the auth tab
            try {
                await chrome.tabs.remove(tabId);
            } catch (e) {
                // Tab might already be closed
            }

            return; // Success, stop polling
        }

        // Auth data not found yet, try again
        setTimeout(() => pollForAuthData(tabId, attempt + 1), pollInterval);

    } catch (error) {
        console.log('[Tube Link Fixer] Could not check auth (tab may be closed):', error.message);
        // If tab is closed, stop polling
        if (error.message.includes('No tab') || error.message.includes('cannot access')) {
            return;
        }
        // Otherwise retry
        setTimeout(() => pollForAuthData(tabId, attempt + 1), pollInterval);
    }
}

// Handle extension install
chrome.runtime.onInstalled.addListener((details) => {
    if (details.reason === 'install') {
        console.log('[Tube Link Fixer] Extension installed');
    } else if (details.reason === 'update') {
        console.log('[Tube Link Fixer] Extension updated');
    }
});
